// Enhanced Form Validation Script - form-validation.js
class FormValidator {
    constructor(formId) {
        this.form = document.getElementById(formId);
        this.init();
    }

    init() {
        if (!this.form) return;
        
        // Add real-time validation
        this.form.addEventListener('submit', (e) => this.validateForm(e));
        
        // Add input event listeners for real-time validation
        const inputs = this.form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', () => this.validateField(input));
            input.addEventListener('input', () => this.clearError(input));
        });

        // Special validation for time fields
        const timeInputs = this.form.querySelectorAll('input[type="time"]');
        timeInputs.forEach(input => {
            input.addEventListener('change', () => this.validateTime(input));
        });

        // Special validation for date fields
        const dateInputs = this.form.querySelectorAll('input[type="date"]');
        dateInputs.forEach(input => {
            input.addEventListener('change', () => this.validateDate(input));
        });
    }

    validateForm(e) {
        e.preventDefault();
        
        let isValid = true;
        const inputs = this.form.querySelectorAll('input[required], select[required], textarea[required]');
        
        inputs.forEach(input => {
            if (!this.validateField(input)) {
                isValid = false;
            }
        });

        // Validate radio groups
        const radioGroups = this.form.querySelectorAll('input[type="radio"][required]');
        const radioGroupsMap = new Map();
        
        radioGroups.forEach(radio => {
            const name = radio.name;
            if (!radioGroupsMap.has(name)) {
                radioGroupsMap.set(name, false);
            }
            if (radio.checked) {
                radioGroupsMap.set(name, true);
            }
        });

        radioGroupsMap.forEach((isChecked, name) => {
            if (!isChecked) {
                const firstRadio = this.form.querySelector(`input[name="${name}"]`);
                this.showError(firstRadio, 'يرجى اختيار أحد الخيارات');
                isValid = false;
            }
        });

        if (isValid) {
            this.showLoading();
            // Submit the form after a short delay to show loading state
            setTimeout(() => {
                this.form.submit();
            }, 1000);
        } else {
            this.showMessage('يرجى تصحيح الأخطاء قبل الإرسال', 'error');
            // Scroll to first error
            const firstError = this.form.querySelector('.error');
            if (firstError) {
                firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
        }
    }

    validateField(field) {
        const value = field.value.trim();
        const fieldName = field.name || field.id;
        
        // Clear previous errors
        this.clearError(field);

        // Required field validation
        if (field.hasAttribute('required') && !value) {
            this.showError(field, 'هذا الحقل مطلوب');
            return false;
        }

        // Email validation
        if (field.type === 'email' && value) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(value)) {
                this.showError(field, 'يرجى إدخال بريد إلكتروني صحيح');
                return false;
            }
        }

        // Number validation
        if (field.type === 'number' && value) {
            if (isNaN(value) || Number(value) < 0) {
                this.showError(field, 'يرجى إدخال رقم صحيح موجب');
                return false;
            }
            
            // Validate specific number ranges
            if (field.name === 'students_in_program' && value > 1000) {
                this.showError(field, 'عدد التلاميذ يجب أن يكون أقل من 1000');
                return false;
            }
            
            if (field.name === 'classes_in_program' && value > 50) {
                this.showError(field, 'عدد الفصول يجب أن يكون أقل من 50');
                return false;
            }
        }

        // Text validation for Arabic names
        if (field.type === 'text' && value) {
            if (field.name.includes('name') && !this.isValidArabicName(value)) {
                this.showError(field, 'يرجى إدخال اسم عربي صحيح');
                return false;
            }
        }

        return true;
    }

    validateTime(field) {
        const value = field.value;
        if (!value) return true;

        const [hours, minutes] = value.split(':').map(Number);
        
        if (hours < 0 || hours > 23 || minutes < 0 || minutes > 59) {
            this.showError(field, 'يرجى إدخال وقت صحيح');
            return false;
        }

        // Validate teaching time order
        if (field.name === 'teaching_start' || field.name === 'teaching_end') {
            const startTime = this.form.querySelector('input[name="teaching_start"]').value;
            const endTime = this.form.querySelector('input[name="teaching_end"]').value;
            
            if (startTime && endTime && startTime >= endTime) {
                this.showError(field, 'وقت البداية يجب أن يكون قبل وقت النهاية');
                return false;
            }
        }

        return true;
    }

    validateDate(field) {
        const value = field.value;
        if (!value) return true;

        const selectedDate = new Date(value);
        const today = new Date();
        
        // Reset time part for comparison
        today.setHours(0, 0, 0, 0);
        selectedDate.setHours(0, 0, 0, 0);

        if (selectedDate > today) {
            this.showError(field, 'لا يمكن اختيار تاريخ في المستقبل');
            return false;
        }

        // Validate visit date is not too old (more than 1 year)
        const oneYearAgo = new Date();
        oneYearAgo.setFullYear(oneYearAgo.getFullYear() - 1);
        
        if (selectedDate < oneYearAgo) {
            this.showError(field, 'التاريخ يجب أن يكون خلال السنة الماضية');
            return false;
        }

        return true;
    }

    isValidArabicName(text) {
        // Basic Arabic character validation
        const arabicRegex = /^[\u0600-\u06FF\s]+$/;
        return arabicRegex.test(text);
    }

    showError(field, message) {
        field.classList.add('error');
        
        let errorDiv = field.parentNode.querySelector('.error-message');
        if (!errorDiv) {
            errorDiv = document.createElement('div');
            errorDiv.className = 'error-message';
            field.parentNode.appendChild(errorDiv);
        }
        
        errorDiv.textContent = message;
        errorDiv.style.display = 'block';
        
        // Add error styling to parent form-group
        const formGroup = field.closest('.form-group');
        if (formGroup) {
            formGroup.classList.add('error');
        }
    }

    clearError(field) {
        field.classList.remove('error');
        const errorDiv = field.parentNode.querySelector('.error-message');
        if (errorDiv) {
            errorDiv.style.display = 'none';
        }
        
        // Remove error styling from parent form-group
        const formGroup = field.closest('.form-group');
        if (formGroup) {
            formGroup.classList.remove('error');
        }
    }

    showLoading() {
        const submitBtn = this.form.querySelector('button[type="submit"]');
        if (!submitBtn) return;
        
        const originalText = submitBtn.innerHTML;
        const originalWidth = submitBtn.offsetWidth;
        
        // Set fixed width to prevent button resizing
        submitBtn.style.width = `${originalWidth}px`;
        
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الحفظ...';
        submitBtn.disabled = true;
        
        // Revert after 5 seconds (safety net)
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
            submitBtn.style.width = '';
        }, 5000);
    }

    showMessage(message, type = 'info') {
        const messageDiv = document.createElement('div');
        messageDiv.className = `validation-message ${type}`;
        messageDiv.innerHTML = `
            <i class="fas fa-${this.getMessageIcon(type)}"></i>
            <span>${message}</span>
        `;
        
        // Add styles if not already added
        this.addMessageStyles();
        
        // Position the message before the form
        this.form.parentNode.insertBefore(messageDiv, this.form);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (messageDiv.parentNode) {
                messageDiv.parentNode.removeChild(messageDiv);
            }
        }, 5000);
    }

    getMessageIcon(type) {
        const icons = {
            error: 'exclamation-circle',
            success: 'check-circle',
            warning: 'exclamation-triangle',
            info: 'info-circle'
        };
        return icons[type] || 'info-circle';
    }

    addMessageStyles() {
        if (document.getElementById('validation-styles')) return;
        
        const styles = `
            <style id="validation-styles">
                .validation-message {
                    padding: 15px 20px;
                    margin: 20px 0;
                    border-radius: 12px;
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    font-weight: 600;
                    animation: slideDown 0.3s ease;
                }
                
                .validation-message.error {
                    background: #fee;
                    border: 2px solid #f56565;
                    color: #c53030;
                }
                
                .validation-message.success {
                    background: #f0fff4;
                    border: 2px solid #48bb78;
                    color: #2d3748;
                }
                
                .validation-message.warning {
                    background: #fffaf0;
                    border: 2px solid #ed8936;
                    color: #dd6b20;
                }
                
                .validation-message.info {
                    background: #ebf8ff;
                    border: 2px solid #4299e1;
                    color: #2b6cb0;
                }
                
                .error-message {
                    color: #e53e3e;
                    font-size: 0.875rem;
                    margin-top: 5px;
                    display: none;
                    font-weight: 500;
                }
                
                .form-group.error input,
                .form-group.error select,
                .form-group.error textarea {
                    border-color: #e53e3e;
                    background-color: #fef5f5;
                }
                
                @keyframes slideDown {
                    from {
                        opacity: 0;
                        transform: translateY(-10px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }
            </style>
        `;
        
        document.head.insertAdjacentHTML('beforeend', styles);
    }
}

// Auto-initialize validation for all forms with IDs
document.addEventListener('DOMContentLoaded', function() {
    // Initialize basic data form
    if (document.getElementById('basicDataForm')) {
        new FormValidator('basicDataForm');
    }
    
    // Initialize teacher 1 form
    if (document.getElementById('teacher1Form')) {
        new FormValidator('teacher1Form');
    }
    
    // Initialize teacher 2 form
    if (document.getElementById('teacher2Form')) {
        new FormValidator('teacher2Form');
    }
    
    // Add any additional form validations here
    console.log('Form validation initialized');
});